<?php
namespace AWeberWebFormPluginNamespace;

#[\AllowDynamicProperties]
class AWeberException extends \Exception { }

/**
 * Thrown when the API returns an error. (HTTP status >= 400)
 *
 *
 * @uses AWeberException
 * @package
 * @version $id$
 */
class AWeberAPIException extends AWeberException {

    public $type;
    public $status;
    public $message;
    public $documentation_url;
    public $url;

    public function __construct($error, $url) {
        // record specific details of the API exception for processing
        $this->url = $url;
        $this->type = is_array($error) && array_key_exists('type', $error) ? $error['type'] : '';
        $this->status = is_array($error) && array_key_exists('status', $error) ? $error['status'] : '';
        $this->message = is_array($error) && array_key_exists('error_description', $error) ? $error['error_description'] : '';
        $this->documentation_url = is_array($error) && array_key_exists('documentation_url', $error) ? $error['documentation_url'] : '';

        parent::__construct($this->message);
    }
}

/**
 * Thrown when attempting to use a resource that is not implemented.
 *
 * @uses AWeberException
 * @package 
 * @version $id$
 */
class AWeberResourceNotImplemented extends AWeberException {

    public function __construct($value) {
        parent::__construct("Resource \"{$value}\" is not implemented on this resource.");
    }
}

/**
 * AWeberMethodNotImplemented
 *
 * Thrown when attempting to call a method that is not implemented for a resource
 * / collection.  Differs from standard method not defined errors, as this will
 * be thrown when the method is infact implemented on the base class, but the
 * current resource type does not provide access to that method (ie calling
 * getByMessageNumber on a web_forms collection).
 *
 * @uses AWeberException
 * @package 
 * @version $id$
 */
class AWeberMethodNotImplemented extends AWeberException {

    public function __construct() {
        parent::__construct("This method is not implemented by the current resource.");

    }
}

/**
 * AWeberOAuthException
 *
 * OAuth exception, as generated by an API JSON error response
 * @uses AWeberException
 * @package 
 * @version $id$
 */
class AWeberOAuthException extends AWeberException {

    public function __construct($type, $message) {
        $this->type = $type;
        $this->message = $message;
        parent::__construct("{$type}: {$message}");
    }
}

/**
 * AWeberOAuthDataMissing
 *
 * Used when a specific piece or pieces of data was not found in the 
 * response. This differs from the exception that might be thrown as
 * an AWeberOAuthException when parameters are not provided because
 * it is not the servers' expectations that were not met, but rather
 * the expecations of the client were not met by the server.
 *
 * @uses AWeberException
 * @package 
 * @version $id$
 */
class AWeberOAuthDataMissing extends AWeberException {

    public function __construct($missing) {
        if (!is_array($missing)) $missing = array($missing); 
        $this->missing = $missing;
        $required = join(', ', $this->missing);
        parent::__construct("OAuthDataMissing: Response was expected to contain: {$required}");

    }
}

/**
 * AWeberResponseError
 *
 * This is raised when the server returns a non-JSON response. This
 * should only occur when there is a server or some type of connectivity
 * issue.
 *
 * @uses AWeberException
 * @package 
 * @version $id$
 */
class AWeberResponseError extends AWeberException {

    public function __construct($uri) {
        $this->uri = $uri;
        parent::__construct("Request for {$uri} did not respond properly.");
    }

}


class AWeberTokenRefreshException extends AWeberException {
    
    public function __construct() {
        $message = 'OAuth2 Token Refresh Exception!';
        $this->status = 401;
        parent::__construct($message);
    }

}


class AWeberOAuth2TokenExpired extends AWeberException {

    public function __construct($error, $status_code) {
        $message = 'OAuth2 token expired!';
        $this->status = $status_code;
        parent::__construct($message);
    }

}

class AWeberOAuth2Exception extends AWeberException {

    public function __construct($error, $status_code) {
        $this->error = $error;
        $this->status = $status_code;
        // Errors can be, invalid_request, invalid_client,
        // invalid_grant, unauthorized_client, unsupported_grant_type
        parent::__construct($this->error);
    }

}
